// --- Sidebar/Event UI and Filtering ---
let dliFilterMode = 'ga_event'; // 'ga_event' or 'all'
window._dli_sidebar_events = [];

function renderSidebarEvents() {
  const list = document.getElementById('dli-sidebar-events');
  if (!list) return;
  list.innerHTML = '';
  let filtered = window._dli_sidebar_events;
  if (dliFilterMode === 'ga_event') {
    filtered = filtered.filter(ev => ev.eventData && ev.eventData.event === 'ga_event');
  }
  if (!filtered.length) {
    list.innerHTML = '<div class="dli-no-events">No events captured yet.</div>';
    return;
  }
  filtered.forEach(ev => {
    const div = document.createElement('div');
    div.className = 'dli-event-card';
    // Remove (A), (INPUT), (BUTTON) etc. from button label
    let btnText = ev.clickedElement?.text || '(unknown)';
    btnText = btnText.replace(/\s*\([^)]+\)\s*$/, '');
    // Pretty-print event fields without curly braces
    let eventFields = '';
    if (ev.eventData && typeof ev.eventData === 'object') {
      for (const [key, value] of Object.entries(ev.eventData)) {
        eventFields += `<div><b>${key}:</b> <span style=\"color:#333;\">${JSON.stringify(value)}</span></div>`;
      }
    }
    div.innerHTML = `
      <div><b>Button:</b> ${btnText}</div>
      <div><b>Event:</b>${eventFields}</div>
    `;
    list.appendChild(div);
  });
}

function addSidebarFilterToggle() {
  const header = document.getElementById('dli-sidebar-header');
  if (!header || document.getElementById('dli-filter-toggle')) return;
  const toggle = document.createElement('button');
  toggle.id = 'dli-filter-toggle';
  toggle.textContent = 'GA Events Only';
  toggle.style.marginLeft = 'auto';
  toggle.style.background = '#fff2';
  toggle.style.color = '#fff';
  toggle.style.border = 'none';
  toggle.style.borderRadius = '6px';
  toggle.style.padding = '0.3em 0.9em';
  toggle.style.fontWeight = '600';
  toggle.style.cursor = 'pointer';
  toggle.onclick = function() {
    dliFilterMode = dliFilterMode === 'ga_event' ? 'all' : 'ga_event';
    toggle.textContent = dliFilterMode === 'ga_event' ? 'GA Events Only' : 'All Events';
    renderSidebarEvents();
  };
  header.appendChild(toggle);
}

function ready(fn) {
  if (document.body) return fn();
  const obs = new MutationObserver(() => {
    if (document.body) {
      obs.disconnect();
      fn();
    }
  });
  obs.observe(document.documentElement, { childList: true });
}

function injectExternalScript() {
  if (document.getElementById('dli-injected-script')) return;
  const script = document.createElement('script');
  script.id = 'dli-injected-script';
  script.src = chrome.runtime.getURL('inject.js');
  script.type = 'text/javascript';
  document.documentElement.appendChild(script);
}

function injectSidebar() {
  if (document.getElementById('dli-sidebar')) return;
  const sidebar = document.createElement('div');
  sidebar.id = 'dli-sidebar';
  sidebar.innerHTML = `
    <style id="dli-sidebar-style">
      #dli-sidebar {
        position: fixed;
        top: 0; right: 0;
        width: 370px;
        height: 100vh;
        background: #fff;
        border-left: 2px solid #e0e0e0;
        box-shadow: -2px 0 12px rgba(0,0,0,0.08);
        z-index: 2147483647;
        font-family: 'Inter', 'Segoe UI', Arial, sans-serif;
        color: #222;
        display: flex;
        flex-direction: column;
      }
      body.dli-sidebar-open {
        margin-right: 370px !important;
        transition: margin-right 0.2s cubic-bezier(.4,0,.2,1);
      }
      #dli-sidebar-header {
        background: linear-gradient(135deg, #7c3aed 0%, #8b5cf6 100%);
        color: #fff;
        font-weight: 700;
        font-size: 1.02rem;
        padding: 0.5em 1em 0.5em 1.2em;
        letter-spacing: -0.01em;
        display: flex;
        align-items: center;
        position: relative;
        justify-content: space-between;
        min-height: 38px;
        box-shadow: 0 1px 4px #0001;
      }
      #dli-sidebar-header-title {
        flex: 1;
        text-align: left;
        font-size: 1.02em;
        font-weight: 700;
        letter-spacing: -0.01em;
        margin-right: 0.7em;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
      }
      #dli-sidebar-close {
        background: none;
        border: none;
        color: #111;
        font-size: 1.3em;
        font-weight: 900;
        cursor: pointer;
        z-index: 2;
        padding: 0 0.2em;
        line-height: 1;
        margin-left: 0.5em;
        height: 32px;
        width: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
      }
      #dli-sidebar-events {
        flex: 1;
        overflow-y: auto;
        padding: 1.2em;
        background: #fafbfc;
      }
      .dli-event-card {
        background: #f3f0ff;
        border-radius: 8px;
        box-shadow: 0 1px 2px #0001;
        padding: 0.8em 1em;
        margin-bottom: 1em;
        font-size: 0.98em;
        word-break: break-word;
      }
      .dli-no-events { color: #888; text-align: center; margin-top: 2em; }
      #dli-sidebar-footer {
        background: #f3f0ff;
        border-top: 1px solid #e0e0e0;
        padding: 0.7em 1.2em 0.7em 1.2em;
        display: flex;
        flex-direction: row;
        align-items: flex-end;
        justify-content: center;
      }
      #dli-copy-indicator {
        position: absolute;
        bottom: 60px;
        right: 1.2em;
        background: #1aaf5d;
        color: #fff;
        font-size: 0.92em;
        font-weight: 700;
        border-radius: 12px;
        padding: 0.18em 0.9em;
        box-shadow: 0 1px 4px #0002;
        display: none;
        z-index: 10;
        pointer-events: none;
        transition: opacity 0.2s;
      }
      #dli-sidebar-footer-row {
        display: flex;
        flex-direction: row;
        align-items: center;
        justify-content: space-between;
        width: 100%;
        gap: 1em;
        position: relative;
      }
      #dli-filter-toggle {
        background: #fff2;
        color: #7c3aed;
        border: none;
        border-radius: 6px;
        padding: 0.38em 1em;
        font-weight: 700;
        font-size: 0.97em;
        cursor: pointer;
        height: 44px;
        min-width: 120px;
        display: flex;
        align-items: center;
        justify-content: center;
        white-space: nowrap;
        transition: background 0.15s, color 0.15s;
      }
      #dli-filter-toggle.active {
        background: #7c3aed;
        color: #fff;
      }
      #dli-sidebar-footer-btns {
        display: flex;
        flex-direction: row;
        gap: 1em;
        align-items: center;
      }
      #dli-sidebar-footer button {
        background: #7c3aed;
        color: #fff;
        border: none;
        border-radius: 6px;
        padding: 0.38em 1em;
        font-weight: 700;
        font-size: 0.97em;
        cursor: pointer;
        transition: background 0.15s;
        min-width: 90px;
        height: 44px;
        display: flex;
        align-items: center;
        justify-content: center;
        white-space: nowrap;
      }
      #dli-sidebar-footer button:hover {
        background: #5b21b6;
      }
    </style>
    <div id="dli-sidebar-header">
      <span id="dli-sidebar-header-title">DataLayer Inspector</span>
      <button id="dli-sidebar-close" title="Close">×</button>
    </div>
    <div id="dli-sidebar-events"><div class="dli-no-events">No events captured yet.</div></div>
    <div id="dli-sidebar-footer">
      <div id="dli-sidebar-footer-row">
        <button id="dli-filter-toggle">GA Events Only</button>
        <div id="dli-sidebar-footer-btns">
          <button id="dli-export-btn">Export</button>
          <button id="dli-copy-btn">Copy</button>
          <span id="dli-copy-indicator">Copied!</span>
        </div>
      </div>
    </div>
  `;
  document.body.appendChild(sidebar);
  document.body.classList.add('dli-sidebar-open');
  document.getElementById('dli-sidebar-close').onclick = function() {
    // Set per-tab storage to false when sidebar is closed
    chrome.runtime.sendMessage({ type: 'DLI_GET_TAB_ID' }, function(response) {
      if (response && response.tabId) {
        const tabKey = 'dli_tab_' + response.tabId;
        const setObj = {};
        setObj[tabKey] = false;
        chrome.storage.local.set(setObj, function() {
          console.log('[DLI Content] Sidebar closed, disabled inspector for this tab.');
        });
      }
    });
    sidebar.remove();
    document.body.classList.remove('dli-sidebar-open');
  };
  // Filter toggle
  const filterToggle = document.getElementById('dli-filter-toggle');
  filterToggle.onclick = function() {
    dliFilterMode = dliFilterMode === 'ga_event' ? 'all' : 'ga_event';
    filterToggle.textContent = dliFilterMode === 'ga_event' ? 'GA Events Only' : 'All Events';
    filterToggle.classList.toggle('active', dliFilterMode === 'ga_event');
    renderSidebarEvents();
  };
  filterToggle.textContent = dliFilterMode === 'ga_event' ? 'GA Events Only' : 'All Events';
  filterToggle.classList.toggle('active', dliFilterMode === 'ga_event');
  // Export and Copy logic
  document.getElementById('dli-export-btn').onclick = function() {
    let filtered = window._dli_sidebar_events;
    if (dliFilterMode === 'ga_event') {
      filtered = filtered.filter(ev => ev.eventData && ev.eventData.event === 'ga_event');
    }
    const output = filtered.map(ev => {
      let btnText = ev.clickedElement?.text || '(unknown)';
      btnText = btnText.replace(/\s*\([^)]+\)\s*$/, '');
      let eventFields = '';
      if (ev.eventData && typeof ev.eventData === 'object') {
        for (const [key, value] of Object.entries(ev.eventData)) {
          eventFields += `${key}: ${JSON.stringify(value)}\n`;
        }
      }
      return `Button: ${btnText}\nEvent:\n${eventFields}`;
    }).join('\n---\n');
    // Get the inspected tab's URL for filename
    chrome.runtime.sendMessage({ type: 'DLI_GET_TAB_URL' }, function(response) {
      let urlPart = 'page';
      if (response && response.url) {
        try {
          const urlObj = new URL(response.url);
          urlPart = urlObj.hostname + (urlObj.pathname ? urlObj.pathname.replace(/\/+$/, '').replace(/\//g, '_') : '');
        } catch (e) {
          urlPart = response.url.replace(/^https?:\/\//, '').replace(/\/$/, '').replace(/[^a-zA-Z0-9._-]/g, '_');
        }
      }
      const filename = `datalayer-events-${urlPart}.txt`;
      const blob = new Blob([output], { type: 'text/plain' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      setTimeout(() => {
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
      }, 100);
    });
  };
  document.getElementById('dli-copy-btn').onclick = function() {
    let filtered = window._dli_sidebar_events;
    if (dliFilterMode === 'ga_event') {
      filtered = filtered.filter(ev => ev.eventData && ev.eventData.event === 'ga_event');
    }
    const output = filtered.map(ev => {
      let btnText = ev.clickedElement?.text || '(unknown)';
      btnText = btnText.replace(/\s*\([^)]+\)\s*$/, '');
      let eventFields = '';
      if (ev.eventData && typeof ev.eventData === 'object') {
        for (const [key, value] of Object.entries(ev.eventData)) {
          eventFields += `${key}: ${JSON.stringify(value)}\n`;
        }
      }
      return `Button: ${btnText}\nEvent:\n${eventFields}`;
    }).join('\n---\n');
    navigator.clipboard.writeText(output).then(() => {
      const indicator = document.getElementById('dli-copy-indicator');
      if (indicator) {
        indicator.style.display = 'inline-block';
        indicator.style.opacity = '1';
        setTimeout(() => { indicator.style.display = 'none'; }, 1500);
      }
    });
  };
}

// Listen for events and update the sidebar
window.addEventListener('DLI_EVENT', function(e) {
  window._dli_sidebar_events.push(e.detail);
  renderSidebarEvents();
});

// On every page load, check chrome.storage.local.dli_global_on. If true, inject inspector.
console.log('[DLI Content] content.js loaded');
chrome.runtime.sendMessage({ type: 'DLI_GET_TAB_ID' }, function(response) {
  if (!response || !response.tabId) {
    console.log('[DLI Content] Could not get tabId from background. Not injecting.');
    return;
  }
  const tabKey = 'dli_tab_' + response.tabId;
  chrome.storage.local.get(tabKey, function(data) {
    const isOn = !!data[tabKey];
    console.log('[DLI Content] TabId:', response.tabId, 'Key:', tabKey, 'Value:', isOn);
    if (isOn) {
      console.log('[DLI Content] Inspector ON for this tab: injecting');
      ready(() => {
        injectExternalScript();
        injectSidebar();
      });
    } else {
      console.log('[DLI Content] Inspector OFF for this tab: not injecting');
    }
  });
}); 